<?php
/**
 * Thrive Themes - https://thrivethemes.com
 *
 * @package thrive-apprentice
 */

use Stripe\Event;
use Stripe\Exception\SignatureVerificationException;
use Stripe\Webhook;
use TVA\Stripe\Connection;
use TVA\Stripe\Events\Generic;
use TVA\Stripe\Request;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Silence is golden!
}

class TVA_Stripe_Controller extends TVA_REST_Controller {

	public $base = 'stripe';

	const THRIVE_KEY = '@#$()%*%$^&*(#@$%@#$%93827456MASDFJIK3245';

	const API_URL = 'https://service-api.thrivethemes.com/stripe/connect';

	public function register_routes() {
		$stripe_connection = Connection::get_instance();
		$webhook_endpoint  = $stripe_connection->get_webhook_endpoint();

		register_rest_route( static::$namespace . static::$version, '/' . $this->base . '/connect_account', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'get_connect_account_link' ],
				'permission_callback' => [ 'TVA_Product', 'has_access' ],
				'args'                => [
					'product_url' => [
						'required' => true,
						'type'     => 'string',
					],
				],
			],
		] );

		register_rest_route( static::$namespace . static::$version, '/' . $this->base . '/api_key', [
			[
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => [ $this, 'save_api_key' ],
				'permission_callback' => [ 'TVA_Product', 'has_access' ],
				'args'                => [
					'api_key' => [
						'required' => true,
						'type'     => 'string',
					],
				],
			],
		] );

		register_rest_route( static::$namespace . static::$version, '/' . $this->base . '/' . $webhook_endpoint, [
			[
				'methods'             => WP_REST_Server::ALLMETHODS,
				'callback'            => [ $this, 'generic_listen' ],
				'permission_callback' => '__return_true',
			],
		] );

		register_rest_route( static::$namespace . static::$version, '/' . $this->base . '/products', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_products' ],
				'permission_callback' => [ 'TVA_Product', 'has_access' ],
				'args'                => [
					'force'     => [
						'required' => false,
						'type'     => 'boolean',
						'default'  => false,
					],
					'test_mode' => [
						'required' => true,
						'type'     => 'boolean',
						'default'  => false,
					],
				],
			],
		] );

		register_rest_route( static::$namespace . static::$version, '/' . $this->base . '/prices', [
			[
				'methods'             => WP_REST_Server::READABLE,
				'callback'            => [ $this, 'get_prices' ],
				'permission_callback' => [ 'TVA_Product', 'has_access' ],
				'args'                => [
					'product_id' => [
						'required' => true,
						'type'     => 'string',
					],
					'test_mode'  => [
						'required' => true,
						'type'     => 'boolean',
						'default'  => false,
					],
				],
			],
		] );
	}

	/**
	 * Get the URL to connect a Stripe account
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_REST_Response
	 */
	public function get_connect_account_link( WP_REST_Request $request ) {
		$product_url = $request->get_param( 'product_url' );

		$data = [
			'site_url' => $product_url,
		];

		$hash = md5( static::THRIVE_KEY . serialize( $data ) . static::THRIVE_KEY );

		$url = add_query_arg( [
			'p' => $hash,
		], static::API_URL );

		$response = wp_remote_post( $url, [
			'body' => $data,
		] );

		if ( wp_remote_retrieve_response_code( $response ) !== 200 ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => __( 'An error occurred. Please contact our support.', 'thrive-apprentice' ) ], 400 );
		}

		$response = wp_remote_retrieve_body( $response );
		$is_valid = filter_var( $response, FILTER_VALIDATE_URL ) !== false;

		if ( ! $is_valid ) {
			$response = json_decode( $response );

			return new WP_REST_Response( $response, 400 );
		}

		return new WP_REST_Response( [ 'success' => $is_valid, 'url' => $response ] );
	}

	/**
	 * Generic endpoint to listen for Stripe webhooks
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_REST_Response
	 */
	public function generic_listen( WP_REST_Request $request ) {
		$stripe_connection   = Connection::get_instance();
		$webhook_secret      = $stripe_connection->get_webhook_secret();
		$stripe_signature    = $request->get_header( 'stripe-signature' );
		$test_webhook_secret = $stripe_connection->get_test_webhook_secret();
		$success             = false;
		$message             = __( 'Invalid signature', 'thrive-apprentice' );

		if ( empty( $webhook_secret ) ) {
			$webhook_secret = $test_webhook_secret;
		}

		if ( $stripe_signature && $webhook_secret ) {
			try {
				$this->handle_stripe_event( $request, $stripe_signature, $webhook_secret );
				$success = true;
				$message = __( 'Event processed', 'thrive-apprentice' );

			} catch ( Exception $e ) {
				// If the webhook secret doesn't work try with the test secret too
				if ( ! $stripe_connection->get_test_mode() ) {
					try {
						$stripe_connection->set_test_mode( true );
						$this->handle_stripe_event( $request, $stripe_signature, $test_webhook_secret );
						$success = true;
						$message = __( 'Event processed', 'thrive-apprentice' );
					} catch ( Exception $e ) {
						$message = $e->getMessage();
					}
				} else {
					$message = $e->getMessage();
				}
			}
		}

		return new WP_REST_Response( [ 'success' => $success, 'message' => $message ] );
	}

	/**
	 * @throws SignatureVerificationException
	 */
	protected function handle_stripe_event( $request, $stripe_signature, $webhook_secret ) {
		$stripe_event = Webhook::constructEvent(
			$request->get_body(),
			$stripe_signature,
			$webhook_secret
		);

		/**
		 * Action triggered when a valid Stripe webhook is received
		 *
		 * @param Event $stripe_event
		 */
		do_action( 'tva_stripe_webhook', $stripe_event );
		do_action( 'tva_stripe_webhook_' . $stripe_event->type, $stripe_event );

		$class_name = Generic::get_class_name( $stripe_event->type );

		if ( class_exists( $class_name ) ) {
			/** @var Generic $event */
			$event = new $class_name( $stripe_event );
			$event->do_action();
		}
	}

	/**
	 * Handle the request to save the API key
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_REST_Response
	 */
	public function save_api_key( WP_REST_Request $request ) {
		$api_key     = $request->get_param( 'api_key' );
		$connection  = Connection::get_instance();
		$dash_api    = $connection->get_dash_api_connection();
		$is_test_key = strpos( $api_key, '_test_' ) !== false;

		$dash_api->set_credentials( [
			'api_key'  => $is_test_key ? '' : $api_key,
			'test_key' => $is_test_key ? $api_key : '',
		] );

		try {
			$response = $connection->ensure_endpoint();
			if ( $response['success'] ) {
				$dash_api->save();
			}
		} catch ( Exception $e ) {
			return new WP_REST_Response( [ 'success' => false, 'message' => $e->getMessage() ], 400 );
		}

		return new WP_REST_Response( $response );
	}


	/**
	 * Handle the request to get all products
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_REST_Response
	 */
	public function get_products( WP_REST_Request $request ) {
		$force     = $request->get_param( 'force' );
		$test_mode = $request->get_param( 'test_mode' );
		$message   = '';
		$products  = [];
		$success   = true;

		try {
			$products = Request::get_all_products( $force, $test_mode );
		} catch ( Exception $e ) {
			$message = $e->getMessage();
			$success = false;
		}

		return new WP_REST_Response( [ 'products' => $products, 'success' => $success, 'message' => $message ] );
	}

	/**
	 * Handle the request to get the prices for a product
	 *
	 * @param WP_REST_Request $request
	 *
	 * @return WP_REST_Response
	 */
	public function get_prices( WP_REST_Request $request ) {
		$product_id = $request->get_param( 'product_id' );
		$test_mode  = $request->get_param( 'test_mode' );

		$prices = Request::get_product_prices( $product_id, $test_mode );

		if ( $prices instanceof WP_Error ) {
			return new WP_REST_Response( [ 'success' => false, 'error' => $prices->get_error_message(), ], 400 );
		}

		return new WP_REST_Response( [ 'success' => true, 'prices' => $prices, ] );
	}
}
