<?php
/**
 * Thrive Themes - https://thrivethemes.com
 *
 * @package thrive-apprentice
 */

namespace TVA\Stripe;

use Stripe\Exception\ApiErrorException;
use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Silence is golden!
}

class Request {
	const PRODUCTS_OPTION_TEST = 'tva_stripe_products_test';

	const PRODUCTS_OPTION = 'tva_stripe_products';

	/**
	 * Get all products from Stripe and cache them
	 *
	 * @param $force
	 * @param $test_mode
	 *
	 * @return array
	 */
	public static function get_all_products( $force = false, $test_mode = false ) {
		$option_name = $test_mode ? static::PRODUCTS_OPTION_TEST : static::PRODUCTS_OPTION;
		$products    = get_option( $option_name, [] );
		if ( empty( $products ) || $force ) {
			$products   = [];
			$connection = Connection::get_instance();
			$connection->set_test_mode( $test_mode );
			$product_client = $connection->get_client()->products;
			static::fetch_all( $product_client, $products );

			//make sure that we don't save classes in the option
			$products = array_map( function ( $product ) {
				$product = $product->jsonSerialize();

				//unset not needed fields
				unset( $product['metadata'], $product['attributes'], $product['object'], $product['images'], $product['package_dimensions'], $product['shippable'] );

				return $product;
			}, $products );

			update_option( $option_name, $products, false );
		}

		return $products;
	}

	/**
	 * Clear the cache for the products
	 *
	 * @return void
	 */
	public static function clear_cache() {
		delete_option( static::PRODUCTS_OPTION );
		delete_option( static::PRODUCTS_OPTION_TEST );
	}

	/**
	 * @throws ApiErrorException
	 */
	public static function get_invoice( $invoice_id ) {
		return Connection::get_instance()->get_client()->invoices->retrieve( $invoice_id );
	}


	public static function get_default_filters() {
		return [
			'limit' => 100,
		];
	}

	/**
	 * Generic function to fetch all objects from Stripe
	 *
	 * @param $client
	 * @param $list
	 * @param $filters
	 *
	 * @return void
	 */
	public static function fetch_all( $client, &$list = [], $filters = [] ) {
		$filters = array_merge( static::get_default_filters(), $filters );
		$filters = array_filter( $filters );

		$response = $client->all( $filters, [
			'stripe_account' => Connection::get_instance()->get_account_id(),
		] );

		$list = array_merge( $list, $response->data );

		if ( $response->has_more ) {
			$filters['starting_after'] = $response->data[ count( $response->data ) - 1 ]->id;
			static::fetch_all( $client, $list, $filters );
		}
	}

	/**
	 * Fetch all prices for a product
	 *
	 * @param $product_id
	 * @param $test_mode
	 *
	 * @return WP_Error|array
	 */
	public static function get_product_prices( $product_id, $test_mode = false ) {
		$prices     = [];
		$connection = Connection::get_instance();
		$connection->set_test_mode( $test_mode );
		$price_client = $connection->get_client()->prices;
		try {
			static::fetch_all( $price_client, $prices, [ 'product' => $product_id ] );
		} catch ( ApiErrorException $e ) {
			return new WP_Error( 'stripe_error', $e->getMessage() );
		}

		return $prices;
	}


}
